/* psk31lx:  PSK31/soundcard  for linux
 * Copyright (C) 1998-2000 Hansi Reiser, dl9rdz (dl9rdz@amsat.org)
 *
 * term-psk31.C
 * text-base terminal for the PSK31 core
 * Version: 2.2
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St. Fifth Floor, Boston,
 * MA 02110, USA.
 *
 */

#include <math.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <termios.h>

#include <curses.h>
#include <ncurses.h>

#include "../config.h"
#include "server/server.h"
#include "modes/psk31-coder.h"

#define L1LINE  0
#define TXSTART 1
#define TXLINES 10
#define L2LINE 11
#define RXSTART 12
#define RXLINES 10
#define L3LINE 22
#define STATSTART 23
#define STATLINES 7


/* Device Path */
static char ptt_path[32] = "/dev/ttyS1";

/* Window' management */
WINDOW *win1, *win2, *win3;
WINDOW *label1, *label2, *label3;

PSK31info rxinfo, txinfo;

/* Waterfall data */
float fftbuf[SAMPLES];

/* Default settings... */
static char callsign[40] = "NOCALL"; /* User's callsign */
static int txtrack = 0;         /* Tx frequency follows rx frequency flag */
static float rxfreq = 0.0, txfreq = 0.0; /* Rx/Tx tone freqs (Hz) */
static int transmit = 0, qpsk = 0, lsb = 0, dcd = 0, afc = 1, phdelta = 0;
static int tuning = 0;

int OPTnew = 0;

static char macro[10][40];

void out_status (int force);
void out_tuning (int);

/* Use this variable to remember original terminal attributes. */
struct termios saved_attributes;


#if 1
void reset_input_mode (void)
{
   tcsetattr (STDIN_FILENO, TCSANOW, &saved_attributes);
}

void set_input_mode (void)
{
   struct termios tattr;

   /* Make sure stdin is a terminal. */
   if (!isatty (STDIN_FILENO))
   {
      fprintf (stderr, "Not a terminal.\n");
      exit (EXIT_FAILURE);
   }

   /* Save the terminal attributes so we can restore them later. */
   tcgetattr (STDIN_FILENO, &saved_attributes);
   atexit (reset_input_mode);
   /* Set the funny terminal modes. */
   tcgetattr (STDIN_FILENO, &tattr);
   tattr.c_lflag &= ~(ICANON | ECHO); /* Clear ICANON and ECHO. */
   tattr.c_cc[VMIN] = 1;
   tattr.c_cc[VTIME] = 0;
   tcsetattr (STDIN_FILENO, TCSAFLUSH, &tattr);
}
#endif

void set_tx_freq (float freq)
{
   txfreq = freq;
   commControl (COMM_TXCH, COMM_FREQ, (int) (txfreq * 100));
}

void set_rx_freq (float freq)
{
   rxfreq = freq;
   commControl (COMM_RXCH, COMM_FREQ, (int) (rxfreq * 100));

   if (txtrack)
   {
      set_tx_freq (rxfreq);
   }
}

void change_freq (float offset)
{
   set_rx_freq (rxfreq + offset);
}

void setmode (int q, int l, int a, int d)
{
   qpsk = q;
   lsb = l;
   dcd = d;
   afc = a;
   commControl (COMM_TXCH, COMM_QPSK, qpsk);
   commControl (COMM_RXCH, COMM_QPSK, qpsk);
   commControl (COMM_TXCH, COMM_LSB, lsb);
   commControl (COMM_RXCH, COMM_LSB, lsb);
   commControl (COMM_RXCH, COMM_AFC, afc);
   commControl (COMM_RXCH, COMM_DCD, dcd);
}

static void readconfig (void)
{
   int cfgerr = 0;
   FILE *fp = NULL, *fpcp = NULL;
   char homecfgfile[256], cfgfile[256], buf[80], *ptr;
   float f;

   /*
    * The ini file in $HOME/ overrides the one in $PKG_DATA_DIR file and
    * a psk31lx.ini in the current directory. So look for $HOME/.psk31lx.ini
    * first.  If not found in $HOME, look in PKG_DATA_DIR/psk31lx.
    * If that works, use it and copy it to $HOME for the next time. If those
    * two failed, look in the current directory. Exit if none is found.
    */
   ptr = getenv ("HOME");

   if (ptr != NULL)
   { /* try $HOME/.psk31lx.ini first */
      strncpy (homecfgfile, ptr, 256);
      strcat (homecfgfile, "/.psk31lx.ini");
      fp = fopen (homecfgfile, "r");
   }

   if (fp == NULL)
   { /* if above open failed, try PKG_DATA_DIR/psk31lx.ini */
      strncpy (cfgfile, PKG_DATA_DIR, 256);

      if (cfgfile[strlen (cfgfile) - 1] != '/')
      {
         strcat (cfgfile, "/");
      }

      strcat (cfgfile, "psk31lx.ini");

      fp = fopen (cfgfile, "r");
      if (fp != NULL)
      {  /* not in $HOME, but it is in PKG_DATA_DIR, so copy it to $HOME */
         /* open homecfgfile for write */
         if ((fpcp = fopen (homecfgfile, "w")) == NULL) 
         {
            fprintf (stderr, "\n Can't create psk31lx.ini in %s\n"
                     "Using the one in %s\n", ptr, PKG_DATA_DIR);               
         }
         else
         {
            /* read fp write to fpcp */
            while (fgets (buf, sizeof (buf), fp) != NULL)
            {
               fputs (buf, fpcp);
            }

            /* close output file and rewind the input file*/
            fclose (fpcp); 
            rewind (fp);
         }
      }
   }

   if (fp == NULL)
   { /* if above two opens failed, try ./psk31lx.ini */
      strncpy (cfgfile, "./psk31lx.ini", 256);
      fp = fopen (cfgfile, "r");
   }

   if (fp == NULL)
   { /* everything failed, so go away */
      fprintf (stderr, "\nConfiguration file %s/psk31lx.ini,\n"
               "~/.psk31lx.ini or ./psk31lx.ini was not found.\n"
               "Did you run \"make install\" as root?\n\n", PKG_DATA_DIR);
      exit (1);
   }

   while (fgets (buf, sizeof (buf), fp) != NULL)
   {
      if (sscanf (buf, "CALL=\"%[^\"]", callsign) == 1)
          /**/;
      else if (sscanf (buf, "FREQ=%f", &f) == 1)
         rxfreq = txfreq = f;
      else if (sscanf (buf, "LSB=%d", &lsb) == 1)
          /**/;
      else if (sscanf (buf, "NET=%d", &txtrack) == 1)
          /**/;
      else if (sscanf (buf, "MACRO1=\"%[^\"]", macro[0]) == 1)
          /**/;
      else if (sscanf (buf, "MACRO2=\"%[^\"]", macro[1]) == 1)
          /**/;
      else if (sscanf (buf, "MACRO3=\"%[^\"]", macro[2]) == 1)
          /**/;
      else if (sscanf (buf, "MACRO4=\"%[^\"]", macro[3]) == 1)
          /**/;
      else if (sscanf (buf, "MACRO5=\"%[^\"]", macro[4]) == 1)
          /**/;
      else if (sscanf (buf, "MACRO6=\"%[^\"]", macro[5]) == 1)
          /**/;
      else if (sscanf (buf, "MACRO7=\"%[^\"]", macro[6]) == 1)
          /**/;
      else if (sscanf (buf, "MACRO8=\"%[^\"]", macro[7]) == 1)
          /**/;
      else if (sscanf (buf, "MACRO9=\"%[^\"]", macro[8]) == 1)
          /**/;
      else if (sscanf (buf, "MACRO10=\"%[^\"]", macro[9]) == 1)
          /**/;
      else if (sscanf (buf, "PTTDEV=\"%[^\"]", ptt_path) == 1)
          /**/;
      else
      {
         fprintf (stderr, "Illegal line in config file%s:\n%s",
                  cfgfile, buf);
         cfgerr = 1;
      }
   }

   fclose (fp);

   if (cfgerr)
      exit (1);
}

void usage_exit (char *argv0)
{
   fprintf (stderr,
            "%s\n\n"
            "Usage: %s [-t serialdevice] "
            "[-v] [-q] [-l]] [-t] [-x] [-h]\n"
            "    -t pathname:   path to PTT device\n"
            "    -v:   print current version number\n"
            "    -q:   Enable QPSK mode\n"
            "    -l:   Invert phase shift for LSB mode\n"
            "    -x:   TX freq follows RX freq\n"
            "    -h:   print this help message\n" "\n", PACKAGE_STRING,
            argv0);
   exit (1);
}

void out_status (int force)
{
   static int oq, ol, odcd, ot, otr, oaf;
   static float orx, otx;
   char buf[128];

   out_tuning (force);

   if (orx != roundf (rxfreq) || otx != txfreq || oq != qpsk || ol != lsb
       || odcd != dcd || ot != transmit || otr != txtrack || oaf != afc)
   {
      force = 1;
   }

   if (!force)
      return;

   sprintf (buf, "  %4s/%3s   DCD:%3s   RX=%6.1f   AFC:%3s   TX=%6.1f   NET:%3s"             "  %2s\n",
            qpsk ? "QPSK" : "BPSK",
            lsb ? "LSB" : "USB",
            dcd ? "ON " : "OFF",
            rxfreq,
            afc ? "ON " : "OFF",
            txfreq, txtrack ? "ON " : "OFF", transmit ? "TX" : "RX");
   orx = roundf (rxfreq);
   otx = txfreq;
   oq = qpsk;
   ol = lsb;
   odcd = dcd;
   ot = transmit;
   otr = txtrack;
   oaf = afc;

   putsyx (win3, 0, 0, buf);
   attron (COLOR_PAIR (3));
   wrefresh (win3);
}

void out_tuning (int force)
{
   static int oldp = -1;
   char wfout[22];
   //float level[] = { 20.0, 15.0, 10.0, 5.0, 2.5 };
   //float level[] = { 15.0, 10.0, 5.0, 3.0, 1.5 };
   float level[] = { 10.0, 5.0, 2.5, 1.25, .625 };
   int offset;
   int l, p, i, j;
   int spaces = 10;
   char vert[] = "|";
   char base[] = "---------------------";

   /* output spectrum analyzer */
   offset = rxfreq / SAMPLE_RATE * SAMPLES - 10;
   l = commGetData (COMM_FFTCH, (char *) fftbuf, sizeof (fftbuf));

   if (l == SAMPLES)
   {
      for (i = 0; i < 5; i++)
      {
         for (j = 0; j < 21; j++)
         {
            if (fftbuf[offset + j] > level[i])
            {
               wfout[j] = '.';
            } else
            {
               wfout[j] = ' ';
            }
         }

         wfout[j] = '\0';
         putsyx (win3, i, spaces, wfout); /* wf data  */
         putsyx (win3, i, spaces + 10, vert); /* vert bar */
      }
   }

   putsyx (win3, i, spaces, base);

   /*output the phase scope */
   if (oldp == -1 || force)
   {
      putsyx (win3, 1, 0, "  .....   ");
      putsyx (win3, 2, 0, " .     .  ");
      putsyx (win3, 3, 0, ".       . ");
      putsyx (win3, 4, 0, " .     .  ");
      putsyx (win3, 5, 0, "  .....   ");
   }

   /* phdelta: 0..255 */
   /* Position:     14 15 0 1 2 ... */
   int pos[][2] = {
      {1, 4}, {1, 5}, {1, 6}, {2, 7}, {2, 7},
      {3, 8}, {4, 7}, {4, 7}, {5, 6}, {5, 5},
      {5, 4}, {5, 3}, {5, 2}, {4, 1}, {4, 1},
      {3, 0}, {2, 1}, {2, 1}, {1, 2}, {1, 3}
   };

   p = (int) ((phdelta / 256.0) * 20 + 0.5); /* 0..19 */

   if (p > 19)
      p = 0;

   if (p != oldp || force)
   {
      putsyx (win3, pos[p][0], pos[p][1], "o");
   }

   if (p != oldp && oldp != -1)
   {
      putsyx (win3, STATSTART + pos[oldp][0], pos[oldp][1], ".");
      oldp = p;
   }
}

void out_macroinfo ()
{
   char buf[128];
   int spaces = 33;

   sprintf (buf, " F1: %-15s  F2: %-15s", macro[0], macro[1]);
   putsyx (win3, 1, spaces, buf);

   sprintf (buf, " F3: %-15s  F4: %-15s", macro[2], macro[3]);
   putsyx (win3, 2, spaces, buf);

   sprintf (buf, " F5: %-15s  F6: %-15s", macro[4], macro[5]);
   putsyx (win3, 3, spaces, buf);

   sprintf (buf, " F7: %-15s  F8: %-15s", macro[6], macro[7]);
   putsyx (win3, 4, spaces, buf);

   sprintf (buf, " F9: %-15s  F10:%-15s", macro[8], macro[9]);
   putsyx (win3, 5, spaces, buf);
}

/*
 * cwsend - If we are not transmitting:
 * turn on PTT, transmit CW data, turn off PTT
 * if PTT is already on:
 *  - switch mode to CW, transmit CW data
 *  - if pttoff==0:  leave PTT enabled, switch back to PSK31 mode
 *  - if pttoff==1:  turn of PTT after finishing transmisson
 */
void cwsend (char *s, int pttoff)
{
   if (tuning)
      tuning = 0;

   commControl (COMM_TXCH, COMM_MODE, MO_CWSEND);

   if (!transmit)
   {
      pttoff = 1;
      commControl (COMM_TXCH, COMM_PTT, PTTON | PTTFORCE);
   }

   commPutData (s, 0);

   if (pttoff)
   {
      fprintf (stderr, "turning off PTT!\n");
      commControl (COMM_TXCH, COMM_PTT, PTTOFF);
   }

   commControl (COMM_TXCH, COMM_MODE, MO_NORMAL);
}

void toggle_tune ()
{
   if (tuning)
   {
      commControl (COMM_TXCH, COMM_PTT, PTTOFF);
      commControl (COMM_TXCH, COMM_MODE, MO_NORMAL);
      tuning = 0;
   } else
   {
      commControl (COMM_TXCH, COMM_MODE, MO_TUNE);

      if (!transmit)
         commControl (COMM_TXCH, COMM_PTT, PTTON | PTTFORCE);

      tuning = 1;
   }
}


/*
 * main function
 */
int main (int argc, char **argv)
{
   int res, c, l, i, exitfl = 0;
   int x1, x2, y1, y2;
   char ch;
   char buf[256], buffer[128];
   fd_set rset, wset, eset;
   struct timeval tm;


   readconfig ();

   /* parse command line arguments */
   while ((c = getopt (argc, argv, "t:vqlxh")) != -1)
   {
      switch (c)
      {
         case 't':
            strncpy (ptt_path, optarg, 20);
            break;
         case 'v':
            printf (PACKAGE_STRING "\n");
            exit (1);
         case 'q':
            qpsk = 1;
            break;
         case 'l':
            lsb = 1;
            break;
         case 'x':
            txtrack = 1;
            break;
         case 'h':
            usage_exit (argv[0]);
            break;
         default:
            fprintf (stderr, "illegal option: %c\n", c);
            usage_exit (argv[0]);
            break;
      }

   }

   x1 = x2 = y1 = y2 = 0;
   srandom (time ((time_t *) 0));

   /* Set the xterm to the right size */
   if ((res = system ("resize -s 30 80")) != 0)
   {
      fprintf (stderr, "main: resize window failed\n");
   }

   if (server_main (ptt_path, PKG_DATA_DIR) == 0)
   {
      fprintf (stderr, "server_main failed\n");
      exit (1);
   }


   /* init screen */
   initscr ();
   start_color ();
   init_pair (1, COLOR_BLACK, COLOR_CYAN);
   init_pair (2, COLOR_WHITE, COLOR_BLUE);
   init_pair (3, COLOR_BLACK, COLOR_CYAN);
   init_pair (4, COLOR_BLACK, COLOR_RED);
   init_pair (5, COLOR_WHITE, COLOR_BLACK);

   curs_set (0);
   cbreak ();
   noecho ();
   intrflush (stdscr, FALSE);
   keypad (stdscr, TRUE);

   win1 = newwin (TXLINES, 80, TXSTART, 0);
   scrollok (win1, TRUE);
   win2 = newwin (RXLINES, 80, RXSTART, 0);
   scrollok (win2, TRUE);
   win3 = newwin (STATLINES, 80, STATSTART, 0);

   label1 = newwin (1, 80, L1LINE, 0);
   label2 = newwin (1, 80, L2LINE, 0);
   label3 = newwin (3, 80, L3LINE, 0);

   wbkgd (label1, COLOR_PAIR (5));
   wbkgd (label2, COLOR_PAIR (5));
   wbkgd (label3, COLOR_PAIR (5));

   wbkgd (win1, COLOR_PAIR (1));
   wbkgd (win2, COLOR_PAIR (2));
   wbkgd (win3, COLOR_PAIR (3));
   refresh();

   putsyx (label1, 0, 35, "Transmit");
   putsyx (label2, 0, 35, "Receive");
   putsyx (label3, 0, 35, "Status");

   nodelay (win1, TRUE);
   nodelay (win2, TRUE);

   out_status (1);
   out_macroinfo ();

   wrefresh (win1);
   wrefresh (win2);
   wrefresh (win3);

   setmode (qpsk, lsb, afc, dcd);
   set_rx_freq (rxfreq);
   set_tx_freq (txfreq);

   /* set dcdlevel to something nice */
   commControl (COMM_RXCH, COMM_DCDLEVEL, 15);

   /* go into main loop */
   while (!exitfl)
   {
      fflush (stdout);
      FD_ZERO (&rset);
      FD_ZERO (&wset);
      FD_ZERO (&eset);
      FD_SET (STDIN_FILENO, &rset);
      tm.tv_sec = 0;
      tm.tv_usec = 50000; /* 50ms */
      res = select (17, &rset, &wset, &eset, &tm);
      if (FD_ISSET (STDIN_FILENO, &rset))
      {
         /* handle keyboard input */
         c = getch ();

         switch (c)
         {
            case 0:
            case 3:
            case 4:
            case 5:
            case 6:
            case 15:
            case 16:
            case 17:
            case 19:
            case 22:
            case 25:
            case 26:
            case 28:
            case 29:
            case 30:
            case 31:
               c = 0; /* ignore */
               break;
            case KEY_F (9):
            case KEY_F (1):
            case KEY_F (2):
            case KEY_F (3):
            case KEY_F (4):
            case KEY_F (5):
            case KEY_F (6):
            case KEY_F (7):
            case KEY_F (8):
            {
               char *txt = macro[c - KEY_F0 - 1];
               getyx (win1, y1, x1);
               putsyx (win1, y1+1, 0, txt);
               commPutData (txt, 0);
               c = 0;
               break;
            }
            case 1:
               /* ^A -> afc on/off */
               setmode (qpsk, lsb, !afc, dcd);
               c = 0;
               break;
            case 2:
               /* ^B -> QPSK <> BPSK */
               setmode (!qpsk, lsb, afc, dcd);
               c = 0;
               break;
            case 7:
            case 9:
            case 10:
            case 13:
               break; /* ok (BEL,TAB,NL,CR) */
            case 8:
            case 127:
            case KEY_BACKSPACE:
               c = '\b';
               break; /* Backspace */
            case 11:
               /* ^K -> txtrack on/off */
               txtrack = !txtrack;
               out_status (0);
               c = 0;
               break;
            case 12:
               /* ^L -> LSB <> USB */
               setmode (qpsk, !lsb, afc, dcd);
               c = 0;
               break;
            case 18: /* ^R -> Receive */
               if (tuning)
                  toggle_tune ();
               else
                  commControl (COMM_TXCH, COMM_PTT, PTTOFF);
               c = 0;
               transmit = 0;
               break;
            case 20: /* ^T -> Transmit */
               if (tuning)
               {
                  commControl (COMM_TXCH, COMM_MODE, MO_NORMAL);
                  tuning = 1;
               }
               else
               {
                  commControl (COMM_TXCH, COMM_PTT, PTTON | PTTFORCE);
               }
               c = 0;
               transmit = 1;
               break;
            case 21:
               /* ^U -> Tune on/off */
               toggle_tune ();
               c = 0;
               break;
            case 23:
               /* ^W -> CWID */
               sprintf (buffer, " de %s", callsign);
               cwsend (buffer, 1);
               c = 0;
               break;
            case KEY_UP:
               change_freq (1);
               c = 0;
               break;
            case KEY_DOWN:
               change_freq (-1);
               c = 0;
               break;
            case KEY_RIGHT:
               change_freq (8);
               c = 0;
               break;
            case KEY_LEFT:
               change_freq (-8);
               c = 0;
               break;
            case 24:
               /* ^X -> clear tx and rx windows */
               wclear (win1);
               wrefresh (win1);
               wclear (win2);
               wrefresh (win2);
               /* and redraw win3 */
               wclear (win3);
               wrefresh (win3);
               out_macroinfo ();

               /* redraw labels */               
               putsyx (label1, 0, 35, "Transmit");
               putsyx (label2, 0, 35, "Receive");
               putsyx (label3, 0, 35, "Status");
               c = 0;
               break;
            case 27:
               /* ESC -> Exit */
               endwin();
               exit (0);
         }
         if (c)
         {
            ch = c & A_CHARTEXT;;
            commPutData (&ch, 1);                  /* send to xmit */
         }

         if (ch > 0 && ch < 256)
         {
#if 1
            if (ch == '\b')                           
            {
               getyx (win1, y1, x1);               /* handle backspace */
               if (x1 == 0 && y1 > 0)              /* beginning of a line? */
               {
                  y1 = y1 - 1;                     /* move up one line */
                  x1 = getmaxx (win1);             /* and to the end */
                  wmove (win1, y1, x1);
                  /* scan backwards for last character */
                  while ((winch (win1) & A_CHARTEXT) == ' ')
                  {
                     x1--;
                     wmove (win1, y1, x1);
                  }                                /* got it so clear it */
                  waddch (win1, ' ');
               }
               else
               {  /* erase previous char */
                  wmove (win1, y1, x1 - 1);
                  waddch (win1, ' ' | COLOR_PAIR (1));
                  wmove (win1, y1, x1 - 1);
                  wrefresh (win1);
               }
            }
            else
            {
               waddch (win1, c | COLOR_PAIR (1));  /* for xmit win */
               wrefresh (win1);
            }
#endif
         }
      }

      /* get status info */
      commGetInfo (COMM_TXCH, &txinfo, sizeof (txinfo));
      commGetInfo (COMM_RXCH, &rxinfo, sizeof (rxinfo));
      rxfreq = 0.01 * rxinfo.freq;
      dcd = rxinfo.dcd;
      phdelta = rxinfo.phdelta;
      out_status (1);

      /* handle echo. We could use a different color? And so we do now. */
      l = commGetData (COMM_ECHOCH, buf, sizeof buf);

      if (l > 0)
      {
         for (i = 0; i < l; i++)
         {
            ch = buf[i]; // & A_CHARTEXT;
#if 1
            if (ch == '\b')
            {                                      /* handle backspace */
               getyx (win2, y2, x2);
               if (x2 == 0 && y2 > 0)
               {
                  y2 = y2 - 1;                     /* move up one line */
                  x2 = getmaxx (win2);             /* and to the end */
                  wmove (win2, y2, x2);
                  /* scan backwards for the last character */
                  while ((winch (win2) & A_CHARTEXT) == ' ')
                  {
                     x2--;
                     wmove (win2, y2, x2);
                  }                                /* got it so clear it */
                  waddch (win2, ' ');
               }
               else
               {  /* erase previous char */
                  wmove (win2, y2, x2 - 1);
                  waddch (win2, ' ' | COLOR_PAIR (2));
                  wmove (win2, y2, x2 - 1);
                  wrefresh (win2);
               }
            }
            else
            {
               waddch (win2, ch | COLOR_PAIR (4));  /* for tx echo */
               wrefresh (win2);
            }
#endif
         }
      }

      /* handle receive */
      l = commGetData (COMM_RXCH, buf, sizeof buf);

      if (l > 0)
      {
         for (int i = 0; i < l; i++)
         {
            waddch (win2, buf[i] | COLOR_PAIR (2));     /* for rx char */
            wrefresh (win2);
         }
      }
   }
}
