// @HEADER
// *****************************************************************************
//            NOX: An Object-Oriented Nonlinear Solver Package
//
// Copyright 2002 NTESS and the NOX contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

// Finite Element Problem Class
/* Provides function (F) and Jacobian evaluations for the following equation
 * via a 1D linear finite element discretization with Epetra objects.
 *
 * d2u
 * --- - k * u**2 = 0
 * dx2
 *
 * subject to @ x=0, u=1
 */

#ifndef _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H
#define _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H

#include "Teuchos_RCP.hpp"

// Forward Declarations
class Epetra_Comm;
class Epetra_Map;
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Flag to tell the evaluate routine what objects to fill
enum FillType {F_ONLY, MATRIX_ONLY, ALL};

// Finite Element Problem Class
class FiniteElementProblem {

public:

  // Constructor
  FiniteElementProblem(int NumGlobalElements, Epetra_Comm& Comm);

  // Destructor
  ~FiniteElementProblem();

  // Evaluates the function (F) and/or the Jacobian using the solution
  // values in solnVector.
  bool evaluate(FillType f, const Epetra_Vector *solnVector,
        Epetra_Vector *rhsVector, Epetra_RowMatrix *matrix);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by the FiniteElementProblem class.
  Teuchos::RCP<Epetra_Vector> getSolution();

  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by the FiniteElementProblem class.
  Teuchos::RCP<Epetra_CrsMatrix> getJacobian();

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraph(Epetra_CrsGraph& AA);

private:

  FillType flag;
  Epetra_Map *StandardMap;
  Epetra_Map *OverlapMap;
  Epetra_Import *Importer;
  Teuchos::RCP<Epetra_Vector> initialSolution;
  Epetra_Vector *rhs;
  Epetra_CrsGraph *AA;
  Teuchos::RCP<Epetra_CrsMatrix> A;
  Epetra_Comm *Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes
  int NumMyElements;      // Number of elements owned by this process
  int NumGlobalElements;  // Total Number of elements

};
#endif




